﻿using Hims.Api.Models;
using Hims.Api.Utilities;

using Hims.Domain.Services;
using Hims.Shared.DataFilters;
using Hims.Shared.EntityModels;
using Hims.Shared.Library.Enums;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using Npgsql;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;

namespace Hims.Api.Controllers
{

    /// inheritdoc   

    [Authorize]
    [Route("api/Ambulance")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class AmbulanceController : BaseController
    {
        private readonly IAmbulanceService ambulanceService;
        private readonly IAuditLogService auditLogServices;


        public AmbulanceController(IAmbulanceService ambulanceServices, IAuditLogService auditLogService)
        {
            this.ambulanceService = ambulanceServices;
            this.auditLogServices = auditLogService;
        }


        [HttpPost]
        [Route("add")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> Add([FromBody] AmbulanceModel model, [FromHeader] LocationHeader location)
        {
            model = (AmbulanceModel)EmptyFilter.Handler(model);
            model.LocationId = Convert.ToInt32(location.LocationId);
            var response = await ambulanceService.AddAsync(model);

            if (response > 0)
            {

                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.Ambulance,
                    LogFrom = (int)AccountType.Administrator,
                    LogDate = DateTime.Now,
                    LogDescription = $"{model.CreatedByName} has been added Ambulance {model.AmbulanceNo}.",
                    LocationId = model.LocationId
                };
                await this.auditLogServices.LogAsync(auditLogModel);
            }

            return this.Success(response);
        }


        [HttpPost]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<AmbulanceModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody] AmbulanceModel model, [FromHeader] LocationHeader location)
        {
            model.LocationId = Convert.ToInt32(location.LocationId);
            var ambulance = await this.ambulanceService.FetchAsync(model);
            return ambulance == null ? this.ServerError() : this.Success(ambulance);
        }

        [HttpPost]
        [Route("update")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody] AmbulanceModel model, [FromHeader] LocationHeader header)
        {
            model = (AmbulanceModel)EmptyFilter.Handler(model);
            var response = await this.ambulanceService.UpdateAsync(model);
            if (response > 0)
            {
                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.Ambulance,
                    LogFrom = (int)AccountType.Administrator,
                    LogDate = DateTime.Now,
                    LogDescription = $"{model.CreatedByName} has been updated ambulance no  {model.AmbulanceNo}.",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLogModel);
            }
            return this.Success(response);
        }

       

        [HttpPut]
        [Route("modify-status")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ModifyStatusAsync([FromBody] AmbulanceModel model, [FromHeader] LocationHeader header)
        {

            var response = await this.ambulanceService.ModifyStatusAsync(model.AmbulanceId, Convert.ToInt32(model.ModifiedBy), model.Active);
            if (response == 0)
            {
                return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.Masters,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.Now,
                LogDescription = $"{model.ModifiedByName} has been updated the  status  to ambulance   {model.AmbulanceNo} {(model.Active ? "'Active'" : "'Inactive'")}.",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success(response);
        }

        
        //[HttpPost]
        //[Route("delete")]
        //[ProducesResponseType(typeof(string), 200)]
        //[ProducesResponseType(409)]
        //[ProducesResponseType(500)]
        //public async Task<ActionResult> DeleteAsync([FromBody] AmbulanceModel model, [FromHeader] LocationHeader header)
        //{
        //    try
        //    {
        //        var ambulance = await this.ambulanceService.FindNameByAmbulanceId(model.AmbulanceId);
        //        var response = await this.ambulanceService.DeleteAsync(model.AmbulanceId);
        //        if (response == 0)
        //        {
        //            return this.ServerError();
        //        }

        //        var auditLogModel = new AuditLogModel
        //        {
        //            AccountId = model.ModifiedBy,
        //            LogTypeId = (int)LogTypes.Masters,
        //            LogFrom = (int)AccountType.Administrator,
        //            LogDate = DateTime.Now,
        //            LogDescription = $"{model.ModifiedByName} has been deleted ambulance no {ambulance}.",
        //            LocationId = Convert.ToInt32(header.LocationId)
        //        };
        //        await this.auditLogServices.LogAsync(auditLogModel);

        //        return this.Success(response);
        //    }
        //    catch (NpgsqlException exception)
        //    {
        //        if (exception.Message.Contains("violates foreign key constraint"))
        //        {
        //            return this.Conflict("ambulance can't be deleted. Please contact Administrator.");
        //        }

        //        return this.ServerError();
        //    }
        //}

        

    }

}
